/* favorites.js
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

/* exported FavoritesMenuButton */

import Clutter from 'gi://Clutter';
import Gio from 'gi://Gio';
import GObject from 'gi://GObject';
import Shell from 'gi://Shell';
import St from 'gi://St';

import { gettext as _ } from 'resource:///org/gnome/shell/extensions/extension.js';

import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';

const FAVORITES_ALL = 1;
const FAVORITES_ICON = 1;
const FAVORITES_TEXT = 2;

const KEY_FAVORITE_APPS = 'favorite-apps';
const KEY_FAVORITES_SCOPE = 'favorites-scope';
const KEY_SHOW_FAVORITES = 'show-favorites';

const PopupFavoriteMenuItem = GObject.registerClass(
class PopupFavoriteMenuItem extends PopupMenu.PopupBaseMenuItem {
    _init(app) {
        super._init();
        this._icon = new St.Icon({
            style_class: 'popup-menu-icon',
            x_align: Clutter.ActorAlign.END,
        });
        this.add_child(this._icon);
        this.label = new St.Label({
            text: app.get_name(),
            y_expand: true,
            y_align: Clutter.ActorAlign.CENTER,
        });
        this.add_child(this.label);
        this.label_actor = this.label;
        this._setIcon(app.get_app_info().get_icon());
        this._app = app;
    }

    _setIcon(icon) {
        if (icon instanceof GObject.Object && GObject.type_is_a(icon, Gio.Icon)) {
            this._icon.gicon = icon;
        } else {
            this._icon.icon_name = icon;
        }
    }

    activate(event) {
        let modifiers = event ? event.get_state() : 0;
        let isCtrlPressed = (modifiers & Clutter.ModifierType.CONTROL_MASK) !== 0;
        let openNewWindow = this._app.can_open_new_window() && this._app.state === Shell.AppState.RUNNING && (isCtrlPressed);
        if (openNewWindow) {
            this._app.open_new_window(-1);
        } else {
            this._app.activate();
        }
        Main.overview.hide();
        super.activate(event);
    }
});

export const FavoritesMenuButton = GObject.registerClass(
class FavoritesMenuButton extends PanelMenu.Button {
    _init(settings, path) {
        super._init(0.5, _('Favorites'), false);
        this.menu.actor.add_style_class_name('panelmanager-menu');
        this._hbox = new St.BoxLayout({ style_class: 'panel-status-indicators-box' });
        this._icon = new St.Icon({
            style_class: 'system-status-icon',
            gicon: Gio.icon_new_for_string(path + '/emblem-favorite-symbolic.svg'),
        });
        this._hbox.add_child(this._icon);
        this._label = new St.Label({
            text: _('Favorites'),
            y_expand: true,
            y_align: Clutter.ActorAlign.CENTER,
        });
        this._hbox.add_child(this._label);
        this.add_child(this._hbox);
        this._settings = settings;
        this._settingsChangedId = [
            this._settings.connect('changed::%s'.format(KEY_FAVORITE_APPS), this._reloadFavorites.bind(this)),
            this._settings.connect('changed::%s'.format(KEY_FAVORITES_SCOPE), this._reloadFavorites.bind(this)),
            this._settings.connect('changed::%s'.format(KEY_SHOW_FAVORITES), this._onChangedShowFavorites.bind(this)),
        ];
        this._onChangedShowFavorites();
        this._appSystem = Shell.AppSystem.get_default();
        this._appSystemChangedId = [
            this._appSystem.connect('app-state-changed', this._reloadFavorites.bind(this)),
            this._appSystem.connect('installed-changed', this._reloadFavorites.bind(this)),
        ]
        this._globalSettingsChangedId = global.settings.connect('changed::%s'.format(KEY_FAVORITE_APPS), this._reloadFavorites.bind(this));
        this._reloadFavorites();
    }

    _onChangedShowFavorites() {
        let showFavorites = this._settings.get_enum(KEY_SHOW_FAVORITES);
        this._icon.visible = showFavorites !== FAVORITES_TEXT;
        this._label.visible = showFavorites !== FAVORITES_ICON;
    }

    _onDestroy() {
        global.settings.disconnect(this._globalSettingsChangedId);
        this._appSystemChangedId.forEach(id => {
            this._appSystem.disconnect(id);
        });
        this._settingsChangedId.forEach(id => {
            this._settings.disconnect(id);
        });
        super._onDestroy();
    }

    _reloadFavorites() {
        this.menu.removeAll();
        let appList = [];
        let favorites = this._settings.get_strv(KEY_FAVORITE_APPS)
        favorites.forEach(desktopFile => {
            let app = this._appSystem.lookup_app(desktopFile);
            if (app) appList.push(app);
        });
        if (this._settings.get_enum(KEY_FAVORITES_SCOPE) === FAVORITES_ALL) {
            let favoriteApps = global.settings.get_strv(KEY_FAVORITE_APPS);
            favoriteApps.forEach(desktopFile => {
                let app = this._appSystem.lookup_app(desktopFile);
                if (app && appList.indexOf(app) < 0) appList.push(app);
            });
        }
        appList.sort((a, b) => a.compare_by_name(b));
        appList.forEach(app => {
            this.menu.addMenuItem(new PopupFavoriteMenuItem(app));
        });
        this.visible = this.menu.numMenuItems > 0;
    }
});
