/* userDialog.js
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

/* exported ModalUserDialog */

const { Clutter, GObject } = imports.gi;

const ExtensionUtils = imports.misc.extensionUtils;

const { MessageDialogContent } = imports.ui.dialog;
const { ModalDialog } = imports.ui.modalDialog;

const _ = ExtensionUtils.gettext;

function getDialogContent(content) {
    let dialogContent = null;
    switch (content) {
    case 'clearRecentDocuments':
        dialogContent = {
            title: _('Clear the list of last used documents?'),
            description: _('If you clear the list of last used documents, you clear the following:\n\u2022 All items from the Places \u2192 Last used documents menu item\n\u2022 All items from the last used documents list in all applications'),
            actions: [
                {
                    label: _('Clear list'),
                    signal: 'empty',
                },
            ],
        };
        break;
    case 'deleteTrashEntry':
        dialogContent = {
            title: _('If you delete an object, it will be permanently lost.'),
            description: value => {
                return _('Are you sure you want to permanently delete »%s« from the trash?').format(value);
            },
            actions: [
                {
                    label: _('Delete Permanently'),
                    signal: 'remove',
                },
            ],
        };
        break;
    case 'ejectVolume':
        dialogContent = {
            title: _('Eject the selected volume?'),
            description: value => {
                return _('Volume: %s').format(value);
            },
            actions: [
                {
                    label: _('Eject'),
                    signal: 'eject',
                },
            ],
        };
        break;
    case 'emptyTrash':
        dialogContent = {
            title: _('Empty all items from Trash?'),
            description: _('All items in the Trash will be permanently deleted.'),
            actions: [
                {
                    label: _('Empty the Trash'),
                    signal: 'empty',
                },
            ],
        };
        break;
    case 'removeRecentDocument':
        dialogContent = {
            title: _('Remove the selected document?'),
            description: value => {
                return _('Are you sure you want to remove »%s« from the list of last used documents?').format(value);
            },
            actions: [
                {
                    label: _('Remove'),
                    signal: 'remove',
                },
            ],
        };
        break;
    case 'restoreTrashEntry':
        dialogContent = {
            title: _('Restore the selected object?'),
            description: value => {
                return _('Are you sure you want to restore »%s« from the trash?').format(value);
            },
            actions: [
                {
                    label: _('Restore'),
                    signal: 'restore',
                },
            ],
        };
        break;
    default: // Nothing to do
    }
    return dialogContent;
}

var ModalUserDialog = GObject.registerClass({
    Signals:{
        'eject': {},
        'empty': {},
        'remove': {},
        'restore': {},
    },
}, class ModalUserDialog extends ModalDialog {
    _init(dialogContent, value) {
        super._init({ styleClass: 'modal-dialog' });
        let { title, description, actions } = getDialogContent(dialogContent);
        this.contentLayout.add_child(new MessageDialogContent({
            title: title,
            description: value ? description(value) : description,
        }));
        this.addButton({
            label: _('Cancel'),
            action: () => {
                this.close();
            },
            key: Clutter.KEY_Escape,
        });
        actions.forEach(action => {
            this.addButton({
                label: action.label,
                action: () => {
                    let signalId = this.connect('closed', () => {
                        this.disconnect(signalId);
                        this.emit(action.signal);
                    });
                    this.close();
                },
            });
        });
    }
});
