/* extension.js
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

/* exported init */

const { Clutter, GLib, GObject, Meta, Shell, St } = imports.gi;

const ExtensionUtils = imports.misc.extensionUtils;

const CtrlAltTab = imports.ui.ctrlAltTab;
const Layout = imports.ui.layout;
const Main = imports.ui.main;
const Overview = imports.ui.overview;

const Tweener = imports.tweener.tweener;

const _ = ExtensionUtils.gettext;
const Me = ExtensionUtils.getCurrentExtension();

const OverviewControls = Me.imports.overviewControls;

const STANDARD_TRAY_ICON_IMPLEMENTATIONS = {
    'bluetooth-applet': 'bluetooth',
    'gnome-volume-control-applet': 'volume', // renamed to gnome-sound-applet
                                             // when moved to control center
    'gnome-sound-applet': 'volume',
    'nm-applet': 'network',
    'gnome-power-manager': 'battery',
    'keyboard': 'keyboard',
    'a11y-keyboard': 'a11y',
    'kbd-scrolllock': 'keyboard',
    'kbd-numlock': 'keyboard',
    'kbd-capslock': 'keyboard',
    'ibus-ui-gtk': 'keyboard'
};

// Offset of the original position from the bottom-right corner
const CONCEALED_WIDTH = 3;
const REVEAL_ANIMATION_TIME = 0.2;
const TEMP_REVEAL_TIME = 2;

const BARRIER_THRESHOLD = 70;
const BARRIER_TIMEOUT = 1000;

class LegacyTray {
    constructor() {
        this.actor = new St.Widget({
            name: 'LegacyTray',
            clip_to_allocation: true,
            layout_manager: new Clutter.BinLayout(),
        });
        this.actor.add_constraint(new Layout.MonitorConstraint({
            primary: true,
            work_area: true,
        }));
        this._slideLayout = new OverviewControls.SlideLayout();
        this._slideLayout.slideDirection = OverviewControls.SlideDirection.LEFT;
        this._slider = new St.Widget({
            x_expand: true,
            y_expand: true,
            x_align: Clutter.ActorAlign.START,
            y_align: Clutter.ActorAlign.END,
            layout_manager: this._slideLayout,
        });
        this.actor.add_child(this._slider);
        this._slider.connect('notify::allocation', this._syncBarrier.bind(this));
        this._box = new St.BoxLayout({ style_class: 'legacy-tray' });
        this._slider.add_child(this._box);
        this._concealHandle = new St.Button({
            style_class: 'legacy-tray-handle',
            /* translators: 'Hide' is a verb */
            accessible_name: _('Hide tray'),
            can_focus: true,
        });
        this._concealHandle.child = new St.Icon({ icon_name: 'go-previous-symbolic' });
        this._box.add_child(this._concealHandle);
        this._iconBox = new St.BoxLayout({ style_class: 'legacy-tray-icon-box' });
        this._box.add_child(this._iconBox);
        this._revealHandle = new St.Button({ style_class: 'legacy-tray-handle' });
        this._revealHandle.child = new St.Icon({ icon_name: 'go-next-symbolic' });
        this._box.add_child(this._revealHandle);
        this._revealHandle.bind_property('visible', this._concealHandle, 'visible', GObject.BindingFlags.BIDIRECTIONAL | GObject.BindingFlags.INVERT_BOOLEAN);
        this._revealHandle.connect('notify::visible', this._sync.bind(this));
        this._revealHandle.connect('notify::hover', this._sync.bind(this));
        this._revealHandle.connect('clicked', () => {
            this._concealHandle.show();
        });
        this._concealHandle.connect('clicked', () => {
            this._revealHandle.show();
        });
        this._horizontalBarrier = null;
        this._pressureBarrier = new Layout.PressureBarrier(BARRIER_THRESHOLD, BARRIER_TIMEOUT, Shell.ActionMode.NORMAL);
        this._pressureBarrier.connect('trigger', () => {
            this._concealHandle.show();
        });
        Main.layoutManager.addChrome(this.actor, { affectsInputRegion: false });
        Main.layoutManager.trackChrome(this._slider, { affectsInputRegion: true });
        Main.ctrlAltTabManager.addGroup(this.actor, _('Status Icons'), 'focus-legacy-systray-symbolic', { sortGroup: CtrlAltTab.SortGroup.BOTTOM });
        this._trayManager = new Shell.TrayManager();
        this._trayManagerChangedId = [
            this._trayManager.connect('tray-icon-added', this._onTrayIconAdded.bind(this)),
            this._trayManager.connect('tray-icon-removed', this._onTrayIconRemoved.bind(this)),
        ];
        this._trayManager.manage_screen(Main.panel);
        this._overviewChangedId = [
            Main.overview.connect('showing', () => {
                Tweener.removeTweens(this._slider);
                Tweener.addTween(this._slider, {
                    opacity: 0,
                    time: Overview.ANIMATION_TIME,
                    transition: 'easeOutQuad',
                });
            }),
            Main.overview.connect('shown', this._sync.bind(this)),
            Main.overview.connect('hiding', () => {
                this._sync();
                Tweener.removeTweens(this._slider);
                Tweener.addTween(this._slider, {
                    opacity: 255,
                    time: Overview.ANIMATION_TIME,
                    transition: 'easeOutQuad',
                });
            }),
        ];
        this._monitorsChangedId = Main.layoutManager.connect('monitors-changed', this._sync.bind(this));
        this._inFullscreenChangedId = global.display.connect('in-fullscreen-changed', this._sync.bind(this));
        this._sessionModeUpdatedId = Main.sessionMode.connect('updated', this._sync.bind(this));
        this._sync();
    }

    _onTrayIconAdded(tm, icon) {
        let wmClass = icon.wm_class ? icon.wm_class.toLowerCase() : '';
        if (STANDARD_TRAY_ICON_IMPLEMENTATIONS[wmClass] !== undefined) return;
        let button = new St.Button({
            style_class: 'legacy-tray-icon',
            child: icon,
            button_mask: St.ButtonMask.ONE | St.ButtonMask.TWO | St.ButtonMask.THREE,
            can_focus: true,
        });
        let app = Shell.WindowTracker.get_default().get_app_from_pid(icon.pid);
        if (!app) app = Shell.AppSystem.get_default().lookup_startup_wmclass(wmClass);
        if (!app) app = Shell.AppSystem.get_default().lookup_desktop_wmclass(wmClass);
        if (app) {
            button.accessible_name = app.get_name();
        } else {
            button.accessible_name = icon.title;
        }
        button.connect('clicked', () => {
            icon.click(Clutter.get_current_event());
        });
        button.connect('key-press-event', () => {
            icon.click(Clutter.get_current_event());
            return Clutter.EVENT_PROPAGATE;
        });
        button.connect('key-focus-in', () => {
            this._concealHandle.show();
        });
        this._iconBox.add_child(button);
        if (!this._concealHandle.visible) {
            this._concealHandle.show();
            this._concealHandelTimeoutId = GLib.timeout_add_seconds(GLib.PRIORITY_DEFAULT, TEMP_REVEAL_TIME, () => {
                this._concealHandelTimeoutId = 0;
                this._concealHandle.hide();
                return GLib.SOURCE_REMOVE;
            });
        }
    }

    _onTrayIconRemoved(tm, icon) {
        if (!this.actor.contains(icon)) return;
        icon.get_parent().destroy();
        this._sync();
    }

    _sync() {
        let allowed = Main.sessionMode.hasNotifications;
        let hasIcons = this._iconBox.get_n_children() > 0;
        let inOverview = Main.overview.visible && !Main.overview.animationInProgress;
        let inFullscreen = Main.layoutManager.primaryMonitor.inFullscreen;
        this.actor.visible = allowed && hasIcons && !inOverview && !inFullscreen;
        if (!hasIcons) this._concealHandle.hide();
        let targetSlide;
        if (this._concealHandle.visible) {
            targetSlide = 1.0;
        } else if (!hasIcons) {
            targetSlide = 0.0;
        } else {
            let [, boxWidth] = this._box.get_preferred_width(-1);
            let [, handleWidth] = this._revealHandle.get_preferred_width(-1);
            if (this._revealHandle.hover) {
                targetSlide = handleWidth / boxWidth;
            } else {
                targetSlide = CONCEALED_WIDTH / boxWidth;
            }
        }
        if (this.actor.visible) {
            Tweener.addTween(this._slideLayout, {
                slideX: targetSlide,
                time: REVEAL_ANIMATION_TIME,
                transition: 'easeOutQuad',
            });
        } else {
            this._slideLayout.slide_x = targetSlide;
            this._unsetBarrier();
        }
    }

    _syncBarrier() {
        let rtl = (this._slider.get_text_direction() === Clutter.TextDirection.RTL);
        let [x, y] = this._slider.get_transformed_position();
        let [w, h] = this._slider.get_transformed_size();
        let x1 = Math.round(x);
        if (rtl) x1 += Math.round(w);
        let x2 = x1;
        let y1 = Math.round(y);
        let y2 = y1 + Math.round(h);
        if (this._horizontalBarrier && this._horizontalBarrier.x1 === x1 && this._horizontalBarrier.y1 === y1 && this._horizontalBarrier.x2 === x2 && this._horizontalBarrier.y2 === y2) return;
        this._unsetBarrier();
        let directions = (rtl ? Meta.BarrierDirection.NEGATIVE_X : Meta.BarrierDirection.POSITIVE_X);
        this._horizontalBarrier = new Meta.Barrier({
            display: global.display,
            x1: x1,
            x2: x2,
            y1: y1,
            y2: y2,
            directions: directions,
        });
        this._pressureBarrier.addBarrier(this._horizontalBarrier);
    }

    _unsetBarrier() {
        if (this._horizontalBarrier === null) return;
        this._pressureBarrier.removeBarrier(this._horizontalBarrier);
        this._horizontalBarrier.destroy();
        this._horizontalBarrier = null;
    }

    destroy() {
        Tweener.removeTweens(this._slideLayout);
        Tweener.removeTweens(this._slider);
        if (this._concealHandelTimeoutId) GLib.source_remove(this._concealHandelTimeoutId);
        if (this._sessionModeUpdatedId) Main.sessionMode.disconnect(this._sessionModeUpdatedId);
        if (this._inFullscreenChangedId) global.display.disconnect(this._inFullscreenChangedId);
        if (this._monitorsChangedId) Main.layoutManager.disconnect(this._monitorsChangedId);
        this._overviewChangedId.forEach(id => {
            Main.overview.disconnect(id);
        });
        this._trayManager.unmanage_screen();
        this._trayManagerChangedId.forEach(id => {
            this._trayManager.disconnect(id);
        });
        this._trayManager = null;
        Main.ctrlAltTabManager.removeGroup(this.actor);
        Main.layoutManager.untrackChrome(this._slider);
        Main.layoutManager.removeChrome(this.actor);
        this._unsetBarrier();
        this._pressureBarrier.destroy();
        this._pressureBarrier = null;
        this.actor.destroy();
    }
}

class Extension {
    constructor(uuid) {
        this._uuid = uuid;
        ExtensionUtils.initTranslations();
    }

    _enabled() {
        this._legacyTray = new LegacyTray();
    }

    enable() {
        if (Main.layoutManager._startingUp) {
            this._startupComplete = Main.layoutManager.connect('startup-complete', () => {
                this._enabled();
                Main.layoutManager.disconnect(this._startupComplete);
            });
        } else {
            this._enabled();
        }
    }

    disable() {
        this._legacyTray.destroy();
        this._legacyTray = null;
    }
}

function init(meta) {
    return new Extension(meta.uuid);
}
