/* prefs.js
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

/* exported init buildPrefsWidget */

const Gettext = imports.gettext;

const { Gio, GLib, GObject, Gtk } = imports.gi;

const ExtensionUtils = imports.misc.extensionUtils;

const Me = ExtensionUtils.getCurrentExtension();

const _ = Gettext.domain(Me.metadata['gettext-domain']).gettext;

function checkAppId(appId) {
    if (appId.indexOf('.desktop') < 0) return false;
    if (GLib.file_test(appId, GLib.FileTest.EXISTS) ||
        GLib.file_test('/usr/share/applications/'+appId, GLib.FileTest.EXISTS) ||
        GLib.file_test(GLib.get_home_dir()+'/.local/share/applications/'+appId, GLib.FileTest.EXISTS) ||
        Glib.find_program_in_path(appId)) return true;
    return false;
}

function getPreferences(group) {
    let prefGroup = null;
    switch (group) {
    case 'generally':
        prefGroup = {
            title: _('Generally'),
            prefs: [
                {
                    type: 'switcher',
                    key: 'enable-hot-corners',
                    label: _('Enable hot corners'),
                    tooltip: _('If true, the activities overview can be accessed by moving the mouse to the top-left corner.'),
                    schema: 'org.gnome.desktop.interface',
                },
                {
                    type: 'combo',
                    key: 'show-activities',
                    label: _('Activities button'),
                    tooltip: _('Select the display mode for the Activities button in the top bar.'),
                    values: {
                        'default': _('Default'),
                        'icon': _('GNOME icon'),
                        'none': _('Not show'),
                    },
                },
                {
                    type: 'switcher',
                    key: 'always-show-universal-access-status',
                    label: _('Always show the Universal Access status icon'),
                    tooltip: _('This key overrides the automatic hiding of the Universal Access status icon when no accessibility features are enabled.'),
                    schema: 'org.gnome.desktop.a11y',
                },
                {
                    type: 'switcher',
                    key: 'always-show-input-sources-status',
                    label: _('Always show the Input Sources status icon'),
                    tooltip: _('This key overrides the automatic hiding of the Input Sources status icon when when only one Input Sources are enabled.'),
                },
                {
                    type: 'switcher',
                    key: 'show-dropdown-arrow',
                    label: _('Dropdown arrow'),
                    tooltip: _('Determines whether or not buttons appear in the top bar with a dropdown arrow.'),
                },
                {
                    type: 'separator',
                },
                {
                    type: 'switcher',
                    key: 'show-dash',
                    label: _('Dash'),
                    tooltip: _('Determines whether the Dash is displayed or not.'),
                },
                {
                    type: 'separator',
                },
                {
                    type: 'switcher',
                    key: 'center-new-windows',
                    label: _('Place new windows in the center'),
                    tooltip: _('When true, the new windows will always be put in the center of the active screen of the monitor.'),
                    schema: 'org.gnome.mutter',
                },
            ],
        };
        break;
    case 'applications':
        prefGroup = {
            title: _('Applications'),
            prefs: [
                {
                    type: 'combo',
                    key: 'show-applications',
                    label: _('Applications button'),
                    tooltip: _('Select the display mode for the Applications button in the top bar.'),
                    values: {
                        'all' : _('Icon and Text'),
                        'icon': _('Icon'),
                        'text': _('Text'),
                    },
                },
                {
                    type: 'label',
                    label: _('To specify which applications will be displayed the GNOME menu editor (alacarte) is required.'),
                },
                {
                    type: 'separator',
                },
                {
                    type: 'switcher',
                    key: 'show-dash-applications',
                    label: _('Show Applications icon on Dash'),
                    tooltip: _('Determines whether the Show Applications icon in the Dash is displayed or not.'),
                },
            ],
        };
        if (GLib.file_test('/usr/bin/alacarte', GLib.FileTest.EXISTS)) prefGroup.prefs.splice(1, 1, {
            type : 'button',
            label: _('Change which applications are shown'),
            tooltip: _('Use the GNOME menu editor to specify which applications will be displayed.'),
            app: 'alacarte',
            appName: _('Main Menu'),
        });
        break;
    case 'places':
        prefGroup = {
            title: _('Places'),
            prefs: [
                {
                    type: 'combo',
                    key: 'show-places',
                    label: _('Places button'),
                    tooltip: _('Select the display mode for the Places button in the top bar.'),
                    values: {
                        'all': _('Icon and Text'),
                        'icon': _('Icon'),
                        'text': _('Text'),
                    },
                },
                {
                    type: 'combo',
                    key: 'bookmarks-position',
                    label: _('Position of Bookmarks'),
                    tooltip: _('Select the position for the bookmarks in the places menu. Custom bookmarks are displayed in a submenu.'),
                    values: {
                         'mainmenu': _('Main Menu'),
                         'submenu': _('Submenu'),
                    },
                },
                {
                    type: 'separator',
                },
                {
                    type: 'switcher',
                    key: 'show-desktop-icons',
                    label: _('Have file manager handle the desktop'),
                    tooltip: _('If set to true, then file manager will draw the icons on the desktop.'),
                    schema: 'org.gnome.desktop.background',
                },
            ],
        };
        break;
    case 'favorites':
        prefGroup = {
            title: _('Favorites'),
            prefs: [
                {
                    type: 'combo',
                    key: 'show-favorites',
                    label: _('Favorites button'),
                    tooltip: _('Select the display mode for the Favorites button in the top bar.'),
                    values: {
                        'all': _('Icon and Text'),
                        'icon': _('Icon'),
                        'text': _('Text'),
                    },
                },
                {
                    type: 'combo',
                    key: 'favorites-scope',
                    label: _('Menu of the Favorites button'),
                    tooltip: _('Change which favorites are shown in the Favorites button menu.'),
                    values: {
                        'own': _('Only own Favorites'),
                        'all': _('All Favorites'),
                    },
                },
                {
                    type: 'favorites',
                    key: 'favorite-apps',
                    label: _('Own Favorites'),
                },
            ],
        };
        break;
    case 'datemenu':
        prefGroup = {
            title: _('Date and Time'),
            prefs: [
                {
                    type: 'switcher',
                    key: 'show-events-icon',
                    label: _('Show day events icon'),
                    tooltip: _('If true, display day events icon, in addition to time.'),
                },
                {
                    type: 'switcher',
                    key: 'clock-show-date',
                    label: _('Show date in clock'),
                    tooltip: _('If true, display date in the clock, in addition to time.'),
                    schema: 'org.gnome.desktop.interface',
                },
                {
                    type: 'switcher',
                    key: 'clock-show-seconds',
                    label: _('Whether the clock shows seconds'),
                    tooltip: _('If true, display seconds in the clock.'),
                    schema: 'org.gnome.desktop.interface',
                },
                {
                    type: 'switcher',
                    key: 'clock-show-weekday',
                    label: _('Show weekday in clock'),
                    tooltip: _('If true, display weekday in the clock, in addition to time.'),
                    schema: 'org.gnome.desktop.interface',
                },
                {
                    type: 'separator',
                },
                {
                    type: 'switcher',
                    key: 'show-weekdate',
                    label: _('Show the week date in the calendar'),
                    tooltip: _('If true, display the ISO week date in the calendar.'),
                    schema: 'org.gnome.desktop.calendar',
                },
            ],
        };
        break;
    case 'systemmenu':
        prefGroup = {
            title: _('System menu'),
            prefs: [
                {
                    type: 'combo',
                    key: 'show-systemmenu',
                    label: _('System menu button'),
                    tooltip: _('Select the display mode for the System menu button in the top bar.'),
                    values: {
                        'default': _('Default'),
                        'icon': _('Shutdown icon')
                    }
                },
                {
                    type: 'switcher',
                    key: 'show-battery-percentage',
                    label: _('Show battery percentage'),
                    tooltip: _('If true, display battery percentage in the status menu, in addition to the icon.'),
                    schema: 'org.gnome.desktop.interface',
                },
                {
                    type: 'separator',
                },
                {
                    type: 'switcher',
                    key: 'always-show-audio-input',
                    label: _('Always show the Audio Input widget'),
                    tooltip: _('This key overrides the automatic hiding of the Audio Input widget when none application is recording audio.'),
                },
                {
                    type: 'switcher',
                    key: 'show-hybrid-sleep',
                    label: _('Show Hybrid Sleep'),
                    tooltip: _('If true, display Hybrid Sleep item in the Shutdown menu.'),
                },
                {
                    type: 'switcher',
                    key: 'show-hibernate',
                    label: _('Show Hibernate'),
                    tooltip: _('If true, display Hibernate item in the Shutdown menu.'),
                },
            ],
        };
        break;
    default: // Nothing to do
    }
    return prefGroup;
}

const PanelManagerPreferences = GObject.registerClass(
class PanelManagerPreferences extends Gtk.Notebook {
    _init() {
        super._init({
            scrollable: true,
            margin: 15,
            width_request: 720,
            height_request: 480,
        });
        this._settings = ExtensionUtils.getSettings();
        this._createPrefGroup(getPreferences('generally'));
        this._createPrefGroup(getPreferences('applications'));
        this._createPrefGroup(getPreferences('places'));
        this._createPrefGroup(getPreferences('favorites'));
        this._createPrefGroup(getPreferences('datemenu'));
        this._createPrefGroup(getPreferences('systemmenu'));
        this.show_all();
    }

    _createPrefGroup(group) {
        let prefGroup = new Gtk.Box({
            orientation: Gtk.Orientation.VERTICAL,
            margin: 15,
            spacing: 15,
        });
        group.prefs.forEach(pref => {
            let prefWidget = this._createPrefWidget(pref);
            if (prefWidget) prefGroup.add(prefWidget);
        });
        this.append_page(prefGroup, new Gtk.Label({ label: group.title }));
    }

    _createPrefWidget(pref) {
        let prefWidget = null;
        let settings = pref.schema ? new Gio.Settings({ schema_id: pref.schema }) : this._settings;
        switch (pref.type) {
        case 'favorites':
            prefWidget = new Gtk.Box({ orientation: Gtk.Orientation.VERTICAL });
            let frame = new Gtk.Frame();
            prefWidget.add(frame);
            let scrolledWindow = new Gtk.ScrolledWindow({ expand: true });
            scrolledWindow.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC);
            frame.add(scrolledWindow);
            let listStore = new Gtk.ListStore();
            listStore.set_column_types([ GObject.TYPE_STRING, GObject.TYPE_STRING ]);
            let treeView = new Gtk.TreeView({ model: listStore });
            scrolledWindow.add(treeView);
            treeView.get_selection().set_mode(Gtk.SelectionMode.SINGLE);
            let columnFavorites = new Gtk.TreeViewColumn({
                title: pref.label,
                expand: true,
                sort_column_id: 1,
            });
            let textRenderer = new Gtk.CellRendererText();
            columnFavorites.pack_start(textRenderer, true);
            columnFavorites.add_attribute(textRenderer, 'text', 1);
            treeView.insert_column(columnFavorites, 0);
            let toolbar = new Gtk.Toolbar();
            toolbar.get_style_context().add_class(Gtk.STYLE_CLASS_INLINE_TOOLBAR);
            prefWidget.add(toolbar);
            let addButton = new Gtk.ToolButton({
                icon_name: 'list-add',
                label: _('Select application'),
                is_important: true,
            });
            addButton.connect('clicked', () => {
                this._favoriteSearch(settings, pref.key);
            });
            toolbar.add(addButton);
            let inputButton = new Gtk.ToolButton({
                icon_name: 'list-add',
                label: _('Desktop file ID'),
                is_important: true,
            });
            inputButton.connect('clicked', () => {
                this._favoriteInput(settings, pref.key);
            });
            toolbar.add(inputButton);
            let delButton = new Gtk.ToolButton({
                icon_name: 'list-remove',
                label: _('Remove Favorite'),
                is_important: true,
            });
            delButton.connect('clicked', () => {
                let [ any, model, iter ] = treeView.get_selection().get_selected();
                strArray = settings.get_strv(pref.key);
                if (any && strArray.length > 0) {
                    let item = listStore.get_value(iter, 0);
                    let index = strArray.map(i => {
                        return i;
                    }).indexOf(item);
                    if (index < 0) return;
                    strArray.splice(index, 1);
                    settings.set_strv(pref.key, strArray);
                }
            });
            toolbar.add(delButton);
            let strArray = [];
            let treeViewSelection = treeView.get_selection();
            let updateListStore = () => {
                strArray = settings.get_strv(pref.key);
                strArray.sort((a, b) => {
                    try { a = Gio.DesktopAppInfo.new(a).get_name(); } catch (e) {}
                    try { b = Gio.DesktopAppInfo.new(b).get_name(); } catch (e) {}
                    return a.toLowerCase() > b.toLowerCase();
                });
                listStore.clear();
                strArray.forEach(app => {
                    if (Gio.DesktopAppInfo.new(app)) {
                        let item = Gio.DesktopAppInfo.new(app).get_name();
                        listStore.set(listStore.append(), [ 0, 1 ], [ app, item ]);
                    }
                });
                treeViewSelection.select_path(Gtk.TreePath.new_from_string(String(0)));
            }
            settings.connect('changed::'+pref.key, updateListStore);
            updateListStore();
            break;
        case 'label':
            prefWidget = new Gtk.Label({
                label: pref.label,
                xalign: 0,
            });
            break;
        case 'separator':
            prefWidget = new Gtk.Separator();
            break;
        default:
            let prefAction = null;
            switch (pref.type) {
            case 'button':
                prefAction = new Gtk.Button({
                    label: pref.appName,
                    valign: Gtk.Align.CENTER,
                    width_request: 240,
                });
                prefAction.connect('clicked', () => {
                    let [success, argv] = GLib.shell_parse_argv(pref.app);
                    GLib.spawn_async(null, argv, null, GLib.SpawnFlags.SEARCH_PATH, null);
                });
                break;
            case 'combo':
                prefAction = new Gtk.ComboBoxText({
                    valign: Gtk.Align.CENTER,
                    width_request: 240,
                });
                for (let value in pref.values) {
                    prefAction.append(value, pref.values[value]);
                }
                prefAction.set_active_id(settings.get_string(pref.key));
                prefAction.connect('changed', () => {
                    settings.set_string(pref.key, prefAction.get_active_id());
                });
                settings.connect('changed::%s'.format(pref.key), () => {
                    prefAction.set_active_id(settings.get_string(pref.key));
                });
                break;
            case 'switcher':
                prefAction = new Gtk.Switch({
                    active: settings.get_boolean(pref.key),
                    valign: Gtk.Align.CENTER,
                });
                settings.bind(pref.key, prefAction, 'active', Gio.SettingsBindFlags.DEFAULT);
                break;
            default: // Nothing to do
            }
            if (prefAction) {
                if (pref.tooltip) prefAction.set_tooltip_text(pref.tooltip);
                prefWidget = new Gtk.Box({ orientation: Gtk.Orientation.HORIZONTAL });
                prefWidget.add(new Gtk.Label({
                    label: pref.label,
                    hexpand: true,
                    xalign: 0,
                }));
                prefWidget.add(prefAction);
            }
        }
        return prefWidget;
    }

    _favoriteDialog() {
        let dialog = new Gtk.Dialog({
            title: _('Add Favorite'),
            modal: true,
            resizable: false,
            transient_for: this.get_toplevel(),
            use_header_bar: true,
        });
        dialog.add_button(_('Cancel'), Gtk.ResponseType.CANCEL);
        return dialog;
    }

    _favoriteInput(schema, key) {
        let dialog = this._favoriteDialog();
        let addButton = dialog.add_button(_('Add'), Gtk.ResponseType.OK);
        addButton.sensitive = 0;
        dialog.set_default_response(Gtk.ResponseType.OK);
        let grid = new Gtk.Grid({
            column_spacing: 15,
            margin: 15,
        });
        dialog.get_content_area().add(grid);
        grid.attach(new Gtk.Label({
            label: _('Desktop file ID:'),
            xalign: 0,
        }), 0, 0, 1, 1);
        let entry = new Gtk.Entry({
            activates_default: true,
            expand: true,
        });
        entry.connect('changed', () => {
            addButton.sensitive = 0;
            let appId = entry.get_text();
            if (checkAppId(appId)) {
                let strArray = schema.get_strv(key);
                if (strArray.indexOf(appId) >= 0) return;
                addButton.sensitive = 1;
            }
        });
        grid.attach(entry, 1, 0, 1, 1);
        dialog.connect('response', (dialog, response_id) => {
            if (response_id === Gtk.ResponseType.OK) {
                let appId = entry.get_text();
                let strArray = schema.get_strv(key);
                strArray.push(appId);
                schema.set_strv(key, strArray);
            }
            dialog.destroy();
        });
        dialog.show_all();
    }

    _favoriteSearch(schema, key) {
        let dialog = this._favoriteDialog();
        let addButton = dialog.add_button(_('Add'), Gtk.ResponseType.OK);
        addButton.sensitive = 0;
        dialog.set_default_response(Gtk.ResponseType.OK);
        let grid = new Gtk.Grid({ margin: 15 });
        dialog.get_content_area().add(grid);
        let appChooser = new Gtk.AppChooserWidget();
        appChooser.set_default_text('');
        appChooser.set_show_all(true);
        appChooser.connect('application-selected', () => {
            addButton.sensitive = 0;
            let appInfo = appChooser.get_app_info();
            if (!appInfo) return;
            let strArray = schema.get_strv(key);
            if (strArray.indexOf(appInfo.get_id()) >= 0) return;
            addButton.sensitive = 1;
        });
        grid.attach(appChooser, 0, 0, 1, 1);
        dialog.connect('response', (dialog, response_id) => {
            if (response_id === Gtk.ResponseType.OK) {
                let appInfo = appChooser.get_app_info();
                let strArray = schema.get_strv(key);
                strArray.push(appInfo.get_id());
                schema.set_strv(key, strArray);
            }
            dialog.destroy();
        });
        dialog.show_all();
    }
});

function init() {
    ExtensionUtils.initTranslations();
}

function buildPrefsWidget() {
    return new PanelManagerPreferences();
}
